<?php
/**
 * Remove User From List
 *
 * @package     AutomatorWP\Integrations\Newsletter\Actions\Remove_User_From_List
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Newsletter_Remove_User_From_List extends AutomatorWP_Integration_Action {

    public $integration = 'newsletter';
    public $action = 'newsletter_remove_user_to_list';

    /**
     * The action result
     *
     * @since 1.0.0
     *
     * @var string $result
     */
    public $result = '';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Remove user from a list', 'automatorwp-pro' ),
            'select_option'     => __( 'Remove user from a <strong>list</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: List. */
            'edit_label'        => sprintf( __( 'Remove user from %1$s', 'automatorwp-pro' ), '{list}' ),
            /* translators: %1$s: List. */
            'log_label'         => sprintf( __( 'Remove user from %1$s', 'automatorwp-pro' ), '{list}' ),
            'options'           => array(
                'list' => array(
                    'from' => 'list',
                    'fields' => array(
                        'list' => array(
                            'name' => __( 'List:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'classes' => 'automatorwp-selector',
                            'options_cb' => array( $this, 'lists_options_cb' ),
                            'default' => 'any'
                        )
                    )
                ),
            ),
        ) );

    }

    /**
     * Get lists options
     *
     * @since 1.0.0
     *
     * @return array
     */
    public function lists_options_cb() {

        $options = array(
            'any' => __( 'any list', 'automatorwp-pro' ),
        );

        if ( class_exists( '\Newsletter' ) ) {

            $lists = \Newsletter::instance()->get_lists();

            if ( ! empty( $lists ) ) {
                foreach ( $lists as $list ) {
                    $options['list_' . $list->id] = $list->name;
                }
            }

        }

        return $options;

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        $this->result = '';

        // Shorthand
        $list_id = $action_options['list'];

        // Bail if not list selected
        if( $list_id === '' ) {
            $this->result = __( 'No list provided.', 'automatorwp-pro' );
            return;
        }

        if ( ! class_exists( '\NewsletterUsers' ) ) {
            $this->result = __( 'Class NewsletterUsers not found.', 'automatorwp-pro' );
            return;
        }

        $user = get_user_by( 'ID', $user_id );

        // Setup the NewsletterUsers instance to save the user data
        $newsletter_users = \NewsletterUsers::instance();

        $subscriber_data = $newsletter_users->get_user( $user->user_email );

        // Subscribe user to the list
        $subscriber_data->$list_id = 0;

        // Setup the NewsletterUsers instance to save the user data
       // $newsletter_users = \NewsletterUsers::instance();
        $subscriber = $newsletter_users->save_user( $subscriber_data );

        if( $subscriber ) {
            $this->result = __( 'User removed from list successfully.', 'automatorwp-pro' );
        } else {
            $this->result = __( 'User not removed from list.', 'automatorwp-pro' );
        }

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        $log_meta['list_id'] = $action_options['list'];

        // Store result
        $log_meta['result'] = $this->result;

        return $log_meta;
    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['list_id'] = array(
            'name' => __( 'List unsubscribed', 'automatorwp-pro' ),
            'desc' => __( 'List unsubscribed.', 'automatorwp-pro' ),
            'type' => 'text',
            'before_field' => array( $this, 'list_display_cb' ),
        );

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;
    }

    /**
     * Callback used to render list on logs
     *
     * @since 1.0.0
     *
     * @param array         $field_args
     * @param CMB2_Field    $field
     */
    public function list_display_cb( $field_args, $field ) {

        $list_id = $field->value();

        $lists = $this->lists_options_cb();

        echo ( isset( $lists[$list_id] ) ? $lists[$list_id] : sprintf( __( 'List with ID %s not found.', 'automatorwp-pro' ), $list_id ) );

    }

}

new AutomatorWP_Newsletter_Remove_User_From_List();